/* -*-C-*-
 ##############################################################################
 #
 # File:        demo/at_tach.c
 # RCS:         $Id: at_tach.c,v 1.7 1998/07/20 22:59:22 ericb Exp $
 # Description: Host program for doing resample at tach
 # Created:     October 20, 1997
 # Language:    C
 # Package:     E1432
 #
 # Copyright (C) 1997 - 1998, Hewlett-Packard Company, all rights reserved.
 #
 ##############################################################################
 #
 # This is a host program that will demonstrate order tracking with resample
 # at tach on a single E1432.  It will display time waveforms, 
 # resampled time waveforms, and order data for all channels.  
 #
 # The equipment required for this demo is:
 # 	1. VXI card cage with MXI interface to host
 #  	2. one E1432/33 with at least 4 input channels and a 
 #	   tachometer option AYD
 #
 # Set the logical address of the E1432 to the address specified in the LA1 
 # defined below (default 8).
 #
 # Connect a tach signal to the Tach1 connector with a frequency range of
 # 1KHz to 20Khz and separate 1 volt PP signal with a frequency range of
 # RPM_LOW/60 to RPM_HIGH/60 to inputs.  
 #
 # The higher the tach signal frequency, the more time is
 # taken processing tach pulses by the CPU; so the performance begins 
 # degrading above 10KHz.  Reducing the number of active input channels
 # (set NUM_CHANS to a lower number than is available from the module)
 # and the blocksize will improve performance.
 #
 # To use the special external triggering mode, uncomment "#define EXT_TRIG".
 # and apply a triggering signal to the Tach2 connector.  The resampling
 # will occur after the trigger signal.  Each block is triggered by the 
 # occurance of this signal.
 #
 # If the module can not keep up in real time, it will skip points
 # and issue a warning about lost points.  If the module can not process
 # tach pulses fast enough, it will overflow the internal raw tach buffer
 # and issue an error.
 #
 # The blocksize is set to NUM_REVS * PPR, or the number of revolutions of
 # the tach times the pulses/rev of the tach signal.  The inputs signals are
 # resampled at the frequency of the signal into the Tach1 connector.
 #
 ##############################################################################
 */
#include <math.h>
#include <stdlib.h>		/* For exit */
#include <stdio.h>		/* For printf */
#include <unistd.h>		/* For sleep */
#include "e1432.h"
#include "xplot.h"
#include "err1432.h"

/*#define TIME*/
/*#define ORDER*/
/*#define EXT_TRIG*/

#define WIDTH           140
#define HEIGHT          80 
#define WARNING_MAX	100

/* Wrap this around all the many function calls which might fail */
#define	DEBUG(s)	s
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (0)
#endif

#define NUM_CHANS	16	/* number of inputs */
#define NUM_REVS	2	/* number of revolutions in a block */
#define RPM_HIGH	4000
#define	RPM_LOW		100
#define PPR		360
#define SAMPLE_MODE	E1432_RESAMP_AT_TACH


int
main(void)
{
    int  i, j, nchan;
    SHORTSIZ16 status;
    LONGSIZ32 count;
    SHORTSIZ16 la = 8;
    SHORTSIZ16 chan_list[18];
    SHORTSIZ16 inputs, tach0, tachs, num_tachs;
#ifdef EXT_TRIG
    SHORTSIZ16 tach1;
#endif
    E1432ID hw;
    float rpm;
    struct e1432_hwconfig cf;
    SHORTSIZ16 error;
    FLOATSIZ32 *data[16];
    FLOATSIZ32 *resampled[16];
    FLOATSIZ32 *freq[16], *freqPtr;
    long blocksize;
    long points, order_points;
    char *plotid[48];
    int row, col, id;  
    char geometry[80];
    char title[80];
    float temp, floatRange, span;
    struct e1432_trailer trailer;
    SHORTSIZ16 warning[WARNING_MAX]; 
    unsigned long warningCount;
    SHORTSIZ16 meas_state;
    char *semabinloc = "/opt/e1432/lib/sema.bin";
    float max_orders, upper_order;
    int sample_mode;

    for(i=0; i<16; i++)
    {
	data[i] = NULL;
	resampled[i] = NULL;
	freq[i] = NULL;
	plotid[i] = NULL;
	plotid[i+16] = NULL;
    }

    if(e1432_init_io_driver()) {
       (void) printf("e1432_init_io_driver() failed\n");
       exit(0);
    }

    CHECK(e1432_print_errors(0));

    /* install the downloadable code */
    (void) printf("Checking for E1432 with firmware at logical address %d ... ",
									la);
    (void) fflush(stdout);
    error = e1432_get_hwconfig(1, &la, &cf);
    CHECK(e1432_print_errors(1));
    if (error)
    {
	(void) printf("Not found.\n\n");
	(void) printf("Installing firmware from %s into E1432 at la %d ... ",
					   semabinloc, la);
	(void) fflush(stdout);
	error = e1432_install(1, &la, 0, semabinloc);
	if (error)
	{
	    (void) printf("\ne1432_install failed and returned error %s\n");
	    exit(0);
	}
	(void) printf("Done.\n");
    }
    else
    {
	(void) printf("Found.\n");
    }
    (void) printf("\n");

    CHECK(e1432_assign_channel_numbers(1, &la, &hw));

    /* Create channel group */
    error = e1432_get_hwconfig(1, &la, &cf);
    if(error) {
        (void) printf("error in e1432_get_hwconfig(): %d\n", error);
        exit(0);
    }
    nchan = (cf.input_chans < NUM_CHANS) ? cf.input_chans : NUM_CHANS;

    (void) printf("Using %d input channels\n", nchan);

    /* Create channel group */
    for(i=0; i<nchan; i++)
       chan_list[i] = E1432_INPUT_CHAN(i+1);

    inputs = e1432_create_channel_group(hw, nchan, chan_list);
    if (inputs >= 0)
    {
	(void) printf("e1432_create_channel_group inputs returned %d\n",
			    inputs);
	return -1;
    }

#if 0
    CHECK(e1432_set_internal_debug(hw, inputs, 0x400)); 
#endif

    sample_mode = SAMPLE_MODE;
    max_orders = PPR / 5.12;
    blocksize = NUM_REVS * PPR;
    order_points = blocksize / 5.12 + 1;
    
    if(blocksize > 4096)
    {
	printf("blocksize required > 4096, lower max_orders or PPR\n");
	exit(1);
    }

    printf("max_orders = %g\n", max_orders);
    printf("order_points = %d\n", order_points);
    points = blocksize;

    upper_order = max_orders;

    span = 5000;

    /* Initialize hardware things */
    CHECK(e1432_set_analog_input(hw, inputs,
				     E1432_INPUT_MODE_VOLT,
				     E1432_INPUT_HIGH_NORMAL,
				     E1432_ANTI_ALIAS_ANALOG_ON,
				     E1432_COUPLING_DC, 2.0));

    CHECK(e1432_set_data_size(hw, inputs, E1432_DATA_SIZE_32));
    CHECK(e1432_set_data_mode(hw, inputs, E1432_DATA_MODE_OVERLAP_BLOCK));
    CHECK(e1432_set_blocksize(hw, inputs, blocksize)); 
    CHECK(e1432_get_blocksize(hw, inputs, &blocksize)); 
    printf("blocksize = %d\n", blocksize);
    CHECK(e1432_set_sample_mode(hw, inputs, sample_mode));

    row = col = 0;
    id = 0;
    temp = (float)(points - 1);

    for(i=0; i < nchan; i++) {
      if(data[i] == NULL) {
        data[i] = (FLOATSIZ32 *)malloc(sizeof(FLOATSIZ32) * points);
        if(!data[i]) {
          (void) printf("Can't malloc data array of %d points\n", points);
          exit(0);
        }
	for(j=0; j < points; j++) {
	   data[i][j] = 0.0;
	}
      }

      if(resampled[i] == NULL) {
        resampled[i] = (FLOATSIZ32 *)malloc(sizeof(FLOATSIZ32) * points);
        if(!resampled[i]) {
          (void) printf("Can't malloc resampled array of %d points\n", points);
          exit(0);
        }
	for(j=0; j < points; j++) {
	   resampled[i][j] = 0.0;
	}
      }

      if(freq[i] == NULL) {
        freq[i] = (FLOATSIZ32 *)malloc(sizeof(FLOATSIZ32) * points);
        if(!freq[i]) {
          (void) printf("Can't malloc freq array of %d points\n", points);
          exit(0);
        }
	for(j=0; j < points; j++) {
	   freq[i][j] = 0.0;
	}
      }

      CHECK(e1432_get_range(hw, chan_list[i], &floatRange));

      if(plotid[id] == NULL) { 
#ifdef TIME
        (void) sprintf( geometry, "%dx%d+%d+%d", WIDTH, HEIGHT,
				(WIDTH + 20) * col, (HEIGHT + 40) * row ); 
	(void) sprintf(title, "Time %d", i + 1);

        plotid[id] = xplot_init_plot(data[i], points, temp, floatRange, 
			-floatRange, GENERIC_TRACE, geometry, title);
#endif
	(void) sprintf( geometry, "%dx%d+%d+%d", WIDTH, HEIGHT,
			(WIDTH + 20) * col, (HEIGHT + 40) * (row  + 1)); 
	(void) sprintf(title, "Resamp %d", i + 1);

        plotid[id+8] = xplot_init_plot(resampled[i], points, temp, floatRange, 
			-floatRange, GENERIC_TRACE, geometry, title);
#ifdef ORDER
	(void) sprintf( geometry, "%dx%d+%d+%d", WIDTH, HEIGHT,
			(WIDTH + 20) * col, (HEIGHT + 40) * (row  + 2)); 
	(void) sprintf(title, "Order %d", i + 1);

        plotid[id+16] = xplot_init_plot(freq[i], order_points, upper_order, 
			0.0, -100.0, FREQ_LOG_MAG_TRACE, geometry, title);
#endif
      }
#ifdef TIME
      xplot_change_yautoscale(plotid[id], 0);
      xplot_set_xscale(plotid[id], (float)0.0, temp);
      xplot_set_yscale(plotid[id], floatRange, -floatRange);
      xplot_change_xlabel(plotid[id], "Samples");
      xplot_change_ylabel(plotid[id], "Volts");
      xplot_repaint(plotid[id]); 
#endif
      xplot_change_yautoscale(plotid[id+8], 0);
      xplot_set_xscale(plotid[id+8], (float)0.0, temp);
      xplot_set_yscale(plotid[id+8], floatRange, -floatRange);
      xplot_change_xlabel(plotid[id+8], "Samples");
      xplot_change_ylabel(plotid[id+8], "Volts");
      xplot_repaint(plotid[id+8]); 
#ifdef ORDER
      xplot_change_yautoscale(plotid[id+16], 0);
      xplot_set_xscale(plotid[id+16], (float)0.0, upper_order);
      xplot_set_yscale(plotid[id+16], 0.0, -100.0);
      xplot_change_xlabel(plotid[id+16], "Orders");
      xplot_change_ylabel(plotid[id+16], "dBV");
      xplot_repaint(plotid[id+16]); 
#endif
        
      col++;
      id++;
      if((col % 8) == 0) {
          col = 0;
          row +=3;
          id += 16;
      }

    }

    tach0 = chan_list[nchan] = E1432_TACH_CHAN(1);
#ifdef EXT_TRIG
    num_tachs = 2;
    tach1 = chan_list[nchan+1] = E1432_TACH_CHAN(2);
#else
    num_tachs = 1;
#endif
    tachs = e1432_create_channel_group(hw, num_tachs, &chan_list[nchan]);
    if (tachs >= 0)
    {
	(void) printf("e1432_create_channel_group for tachs returned %d\n",
			    						tachs);
	exit(0);
    }

    CHECK(e1432_set_tach_ppr(hw, tach0, PPR)); 
    CHECK(e1432_set_span(hw, inputs, span));
    CHECK(e1432_get_span(hw, inputs, &span));
    printf("span = %g\n", span);

    CHECK(e1432_set_trigger_level(hw, tachs, E1432_TRIGGER_LEVEL_LOWER, 0.4));
    CHECK(e1432_set_trigger_level(hw, tachs, E1432_TRIGGER_LEVEL_UPPER, 0.5));

    CHECK(e1432_set_max_order(hw, tachs, max_orders)); 
    CHECK(e1432_set_rpm_low(hw, tachs, RPM_LOW)); 
    CHECK(e1432_set_rpm_high(hw, tachs, RPM_HIGH)); 

#ifdef ORDER
    CHECK(e1432_set_calc_data(hw, inputs, E1432_DATA_ORDER));  
#else
    CHECK(e1432_set_calc_data(hw, inputs, E1432_DATA_RESAMP_TIME));  
#endif
    CHECK(e1432_set_window(hw, inputs, E1432_WINDOW_UNIFORM));  
    CHECK(e1432_set_decimation_output(hw, inputs, E1432_MULTIPASS)); 
    CHECK(e1432_set_decimation_oversample(hw, inputs, 
					  E1432_DECIMATION_OVERSAMPLE_ON));

#ifdef EXT_TRIG
    CHECK(e1432_set_trigger_channel(hw, tach1, E1432_CHANNEL_ON));
    CHECK(e1432_set_trigger_slope(hw, tach1, E1432_TRIGGER_SLOPE_NEG));
#else
    CHECK(e1432_set_trigger_channel(hw, tach0, E1432_CHANNEL_ON));
#endif

    CHECK(e1432_set_arm_channel(hw, tach0, E1432_CHANNEL_ON));

#if 1
    CHECK(e1432_set_trigger_delay(hw, tachs, -blocksize/2)); 
#else
    CHECK(e1432_set_trigger_delay(hw, tachs, 0)); 
#endif

#ifndef TIME
    CHECK(e1432_set_enable(hw, inputs,
				E1432_ENABLE_TYPE_TIME, E1432_ENABLE_OFF));
#endif

#ifndef ORDER
    CHECK(e1432_set_enable(hw, inputs,
				E1432_ENABLE_TYPE_ORDER, E1432_ENABLE_OFF));
#endif

    /* Start measurement */
    CHECK(e1432_init_measure(hw, inputs));

    (void) sleep(1);
    CHECK(e1432_get_current_rpm(hw, tach0, &rpm));
    (void) printf("initial current rpm = %g\n", rpm);

    for (;;)
    {
	do  /* Wait for block available and check for errors and warnings  */
	{
            CHECK(e1432_read_register(hw, tach0, 	
					E1432_IRQ_STATUS2_REG, &status));
	    if(status & E1432_IRQ_MEAS_ERROR)
	    {
		if(status & E1432_STATUS2_TACH_OVERFLOW)
		    (void) printf("Tach buffer overflowed\n");
		else
		    (void) printf("Fifo overflowed\n");
		exit(0);
	    }


	    CHECK(e1432_get_meas_state(hw, tach0, &meas_state));
	    if(meas_state == E1432_MEAS_STATE_TESTED)
	    {
		(void) printf("Measurement finished.\n");
		exit(0);
	    }

	    if(status & E1432_IRQ_MEAS_WARNING)
	    {
		/* read out all measurement warnings */
		while(status & E1432_IRQ_MEAS_WARNING)
		{
		    CHECK(e1432_get_meas_warning(hw, inputs, warning, 
						WARNING_MAX, &warningCount));
	   
		    if(warningCount)
		    {
	       		(void) printf("%d Warning", warningCount);
			if(warningCount > 1) printf("s");
			(void) printf(":\n");
		    }

           	    for(i=0; i < warningCount; i++)
	   	    {
			printf("    %s\n", 
					e1432_get_warning_string(warning[i]));
	   	    }
                    CHECK(e1432_read_register(hw, tach0, 	
					E1432_IRQ_STATUS2_REG, &status));
		}
	    }

	}while(e1432_block_available(hw, tach0) == 0); 


#ifdef TIME
	for(i=0; i < nchan; i++) {	/* read time data */

            error = e1432_read_float32_data(hw, chan_list[i],
				E1432_TIME_DATA, data[i], points, 
				&trailer, &count);
            if(error) {
   	      (void) printf("ERROR: e1432_read_float32_data had error = %d\n",
								error);
            }
	}
#endif

        for(i=0; i < nchan; i++) {
	    error = e1432_read_float32_data(hw, chan_list[i], 
			E1432_RESAMP_DATA, resampled[i], points, 
			&trailer, &count);
            if(error) {
   	      (void) printf("ERROR: e1432_read_float32_data had error = %d\n",
								error);
            }
	}

	id = 0;
        for(i=0; i < nchan; i++) {
#ifdef ORDER
	    error = e1432_read_float32_data(hw, chan_list[i], 
			E1432_ORDER_DATA, freq[i], points, 
			&trailer, &count);
            if(error) {
   	      (void) printf("ERROR: e1432_read_float32_data had error = %d\n",
								error);
            }

	    /* make the 0 dB point full scale */
            freqPtr = freq[i];
	    for(j=0; j < points; j++)
		*freqPtr++ /= floatRange;

            freqPtr = freq[i];
	    array_magsq(freqPtr, points/2);
	    array_log(freqPtr, points/2, 10.0);
#endif

#ifdef TIME
            xplot_check_events(plotid[id]);
            xplot_data_update(plotid[id]);
#endif
            xplot_check_events(plotid[id+8]);
            xplot_data_update(plotid[id+8]);    
#ifdef ORDER
            xplot_check_events(plotid[id+16]);
            xplot_data_update(plotid[id+16]);    
#endif
	    if((++id%8)==0) id += 16;
        }
    }
    /*NOTREACHED*/
    return 0;
}

